/**
* \file: errmemd_backend_dlt.c
*
* Definition of  backend to use the dlt component as outptut medium.
* 
* This file implements the interface functions available to handle
* the dlt component as a backend. 
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <syslog.h>
#include <dlt/dlt.h>

#include "errmem_backend.h"

#ifndef _DLT_PACKAGE_MAJOR_VERSION
#define _DLT_PACKAGE_MAJOR_VERSION 1
#define _DLT_PACKAGE_MINOR_VERSION 0
#endif // ifndef _DLT_PACKAGE_MAJOR_VERSION

DLT_DECLARE_CONTEXT(errmem_dlt)

/*
 * Implementation of an error memory backend that dumps error memory
 * messages to Dlt.
 */

typedef struct ErrmemBackendDlt {
	ErrmemBackend_t backend; // error memory backend interface
	int32_t type;            // type of backend "storage"
} ErrmemBackendDlt_t;

static unsigned get_type(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendDlt_t*)backend)->type;
	return 0;
}

// info - show information about this backend
static void info(struct ErrmemBackend* backend, int fd)
{
	(void) backend;
	dprintf(fd, "error memory daemon has created interface to DLT\n");
}

// store - store an error memory message
static unsigned store(
	struct ErrmemBackend* backend, int len,
	struct errmem_message* msg)
{

	unsigned char level = DLT_LOG_INFO;
	unsigned char* message = msg->message;
	char vmmessage[VM_STRING_SZ_MAX];
	unsigned owner;

	(void) backend;
	(void) len;

	if (*message == 0) {
		level = *(message + 4);
		message += 5;
	}

	 memset(vmmessage,'\0', sizeof(char)* VM_STRING_SZ_MAX);

	/* PRQA: Lint Message 648: We explicitly want to create a mask*/
	/*lint -save -e648 */
	owner = ERRMEM_GET_VM(msg->internal.flags);
	/*lint -restore */
	/* log the errmemd message to the viewer */
	if ((msg->type != ERRMEM_TYPE_TRACE) && owner) {
		if(owner == OWNER_VMM)
			snprintf(vmmessage, VM_STRING_SZ_MAX,
					VMM_STRING"%s",
					msg->message);
		else
			snprintf(vmmessage, VM_STRING_SZ_MAX,
					VM_STRING"%s",
					(owner - 2), msg->message);

		DLT_LOG(errmem_dlt, (DltLogLevelType) level,
					DLT_STRING((const char *) vmmessage));
	} else
		DLT_LOG(errmem_dlt, (DltLogLevelType) level,
			DLT_STRING((const char *) message));

	return msg->internal.seqnum;
}

// destroy - free resources
static void destroy(struct ErrmemBackend* backend)
{
	ErrmemBackendDlt_t* dev = (ErrmemBackendDlt_t*) backend;

	DLT_UNREGISTER_CONTEXT(errmem_dlt);
	DLT_UNREGISTER_APP();

	if (dev)
		free(dev);
}

// errmem_backend_create_dlt - create a DLT error memory backend
ErrmemBackend_t* create_dlt_structure(ErrmemBackendDev_t* p_dev)
{
	ErrmemBackendDlt_t* dev = NULL;

	if (p_dev) {
		dev = (ErrmemBackendDlt_t*)get_mem(sizeof(ErrmemBackendDlt_t));

		if (!dev)
			syslog(LOG_CRIT, "%s %s",
				   "DLT-Backend could not be created - error: ",
				   strerror(ENOMEM));	
		else {
			dev->backend.info     = info;
			dev->backend.store    = store;
			dev->backend.destroy  = destroy;
			dev->backend.get_type = get_type;
			dev->type             = p_dev->type;		

			DLT_REGISTER_APP("ERRMEM", "Error memory daemon");
			DLT_REGISTER_CONTEXT(errmem_dlt, "ERRMEM", "Error memory daemon");
		}
	} else {
		syslog(LOG_CRIT, "%s %s",
			   "DLT-Backend: Parameter check failed - create_dlt_structure",
			   strerror(EINVAL));
	}	

	if (dev)
		return &dev->backend;
	else
		return NULL;
}
